/*******************************************************************************
FILE NAME: servo.c

Copyright 1999 SAGE Telecommunications Pty Ltd
All rights Reserved
The information contained herein is confidential property of the company. The
use, copying, transfer or disclosure of such information is prohibited except
by express written agreement with the Company.

AUTHOR: Rod Egan from SAGE Telecommunications Pty Ltd
                       84 Vickers St
                       Hamersley WA 6022
                   ph (08) 9344-8474 fax (08) 9344-3603
DESCRIPTION: 8 channel servo controller with serial interface


REVISION HISTORY:
        0.0    26-Sep-99    initial source
        0.1     5-Dec-99    added dual baud rate, increased resolution & changed cpu to 16c711
*******************************************************************************/


//#include <16F84.H>
#include <16C711.H>
#include "picreg.h"
#fuses HS,NOPROTECT,WDT
#zero_ram
#define CPU_CLOCK 4000000

#use delay(clock=CPU_CLOCK)

/* We divide the o/p pulse into three discrete time intervals due to the limitations of the counter.
    the first interval is 1mSec less ~20usec with the o/p active
    The second is that actual 'control pulse' with the o/p still active. from 20uSec till 1.02msec
    The third interval is the remainder of the control pulse period form 20uSec to 1.02mSec
    Giving a total on period of 1-2mSec and an interval of ~20usec between pulses.
*/

// servo zero time reload value
#define SERVO_ZERO_RELOAD 250
// min inter servo delay period
#define SERVO_INTER_RELOAD 5
// servo centre time preset value
#define SERVO_CENTRE_PRELOAD (255/2)



#define RTCC_DIV RTCC_DIV_4


#define hold_cycles 5


#define BUFFER_SIZE 8
#define BUFFER_SIZE_MASK 0x07

#define START_CHAR 0xff


#use FAST_IO( A )
#use FAST_IO( B )

struct io_porta_map
{
           boolean baud_rate;        // bit 0, spare
           boolean spare_a1;         // bit 1, spare
           boolean rx;               // bit 2, rx data i/p
           boolean tx;               // bit 3, tx data o/p
           boolean spare_a4;         // bit 4, spare
} io_porta;

#byte io_porta = PORT_A
#byte io_portb = PORT_B



/* global variable storage */

byte servo_on[ BUFFER_SIZE ];
byte servo_off[ BUFFER_SIZE ];
byte servo_base[ BUFFER_SIZE ];

/* local variable storage */

/* global prototypes declaration */

/* local prototypes declaration */
static int get_char( void );
static int getc_2400( void );
static int getc_9600( void );

/****************************************************************************/
/* rtcc interrupt handler                                                   */
/*  This routine is 'tweaked' to try and ensure consistant timing and speed */
/*                                                                          */
/*  entry:                                                                  */
/*  exit:                                                                   */
/****************************************************************************/
#INT_GLOBAL
void rtcc_interrupt( void )
{
    static int w_temp;
    static int status_temp;
    static int reload = SERVO_ZERO_RELOAD;
    static int index = 1;
    static int mask = 0x01;
    static short mode = TRUE;
    static short start_pulse = TRUE;

#asm
    movwf   w_temp
    swapf   STATUS,W
    bcf     RP0
    movwf   status_temp
#endasm

    set_rtcc( reload );                 // load preset value

    if( start_pulse )
    {
        if( !mode )
        {
            reload = servo_off[ index ];    // preset off reload value
            index++;                        // index to next servo data
            index &= BUFFER_SIZE_MASK;      // make sure index doesn't overflow            
            mode = TRUE;
        }
        else
        {
            io_portb = 0;                   // servo off
            rotate_left( &mask, 1 );        // move servo mask
            reload = servo_base[ index ];   // preset base reload value
            mode = FALSE;
            start_pulse = FALSE;
        }
    }
    else
    {                                       // int at end of first period
        io_portb = mask;                // servo on        
        start_pulse = TRUE;
        reload = servo_on[ index ];     // preset on reload value        
    }
#asm
    swapf   status_temp,W
    movwf   STATUS
    swapf   w_temp,F
    swapf   w_temp,W
    bcf     T0IF
#endasm

}

/****************************************************************************/
/* serial port routines                                                     */
/*                                                                          */
/*  entry:                                                                  */
/*  exit:                                                                   */
/****************************************************************************/
static int get_char( void )
{
    if( io_porta.baud_rate )
        return( getc_9600() );
    else
        return( getc_2400() );
}

/****************************************************************************/
/* 2400 serial port routines                                                */
/*                                                                          */
/*  entry:                                                                  */
/*  exit:                                                                   */
/****************************************************************************/
#USE RS232( BAUD = 2400, XMIT = PIN_A3, RCV = PIN_A2, RESTART_WDT )
static int getc_2400( void )
{
    return( getc() );
}

/****************************************************************************/
/* 9600 serial port routines                                                */
/*                                                                          */
/*  entry:                                                                  */
/*  exit:                                                                   */
/****************************************************************************/
#USE RS232( BAUD = 9600, XMIT = PIN_A3, RCV = PIN_A2, RESTART_WDT )
static int getc_9600( void )
{
    return( getc() );
}

/****************************************************************************/
/* main                                                                     */
/*                                                                          */
/*  entry:                                                                  */
/*  exit:                                                                   */
/****************************************************************************/
main()
{
    long timeout;
    int servo_index;
    int desired;
    int temp_on, temp_off, temp_base;
    
    setup_counters( RTCC_INTERNAL, RTCC_DIV );

    set_tris_a( 0b00010111 );   //tx o/p, rest i/p
    set_tris_b( 0b00000000 );   // all o/p

    disable_interrupts( GLOBAL );

    set_rtcc( -SERVO_ZERO_RELOAD );
    enable_interrupts( RTCC_ZERO );
    enable_interrupts( GLOBAL );


    for( servo_index = 0; servo_index < sizeof(servo_on); servo_index++ )
    {
        servo_on[ servo_index ] = ~( SERVO_CENTRE_PRELOAD );
        servo_off[ servo_index ] = ~( 255 - SERVO_CENTRE_PRELOAD );
        servo_base[ servo_index ] = ~( SERVO_ZERO_RELOAD );
    }


    while( TRUE )
    {
        restart_wdt();
        if( kbhit() )
        {
            if( get_char() == START_CHAR )      // wait for sync char
            {
                timeout = 0xff;
                while( --timeout )              // wait for servo select
                {
                    restart_wdt();
                    if( kbhit() )
                    {
                        servo_index = get_char();
                        if( servo_index == 255 )        // abort if 255
                            break;
                        servo_index &= BUFFER_SIZE_MASK;
                        timeout = 0xff;
                        while( --timeout )
                        {
                            restart_wdt();
                            if( kbhit() )       // wait for servo value
                            {
                                desired = get_char();
                                if( desired == 255 )        // abort if 255
                                    break;
                                if( desired < 20 )
                                {
                                    temp_base = ~( SERVO_ZERO_RELOAD - 20 );
                                    temp_on = ~( desired + 20 );
                                }
                                else
                                {
                                    temp_base = ~( SERVO_ZERO_RELOAD );                                
                                    temp_on = ~( desired );
                                }
                                if( desired > 235 )
                                    temp_off = ~( 255 - 235 );
                                else
                                    temp_off = ~( 255 - desired );
                                switch( servo_index )
                                {
                                    case 0:
                                        disable_interrupts( GLOBAL );
                                        servo_on[ 0 ] = temp_on;
                                        servo_off[ 0 ] = temp_off;
                                        servo_base[ 0 ] = temp_base;
                                        enable_interrupts( GLOBAL );
                                        break;
                                    case 1:
                                        disable_interrupts( GLOBAL );
                                        servo_on[ 1 ] = temp_on;
                                        servo_off[ 1 ] = temp_off;
                                        servo_base[ 1 ] = temp_base;
                                        enable_interrupts( GLOBAL );
                                        break;
                                    case 2:
                                        disable_interrupts( GLOBAL );
                                        servo_on[ 2 ] = temp_on;
                                        servo_off[ 2 ] = temp_off;
                                        servo_base[ 2 ] = temp_base;
                                        enable_interrupts( GLOBAL );
                                        break;
                                    case 3:
                                        disable_interrupts( GLOBAL );
                                        servo_on[ 3 ] = temp_on;
                                        servo_off[ 3 ] = temp_off;
                                        servo_base[ 3 ] = temp_base;
                                        enable_interrupts( GLOBAL );
                                        break;
                                    case 4:
                                        disable_interrupts( GLOBAL );
                                        servo_on[ 4 ] = temp_on;
                                        servo_off[ 4 ] = temp_off;
                                        servo_base[ 4 ] = temp_base;
                                        enable_interrupts( GLOBAL );
                                        break;
                                    case 5:
                                        disable_interrupts( GLOBAL );
                                        servo_on[ 5 ] = temp_on;
                                        servo_off[ 5 ] = temp_off;
                                        servo_base[ 5 ] = temp_base;
                                        enable_interrupts( GLOBAL );
                                        break;
                                    case 6:
                                        disable_interrupts( GLOBAL );
                                        servo_on[ 6 ] = temp_on;
                                        servo_off[ 6 ] = temp_off;
                                        servo_base[ 6 ] = temp_base;
                                        enable_interrupts( GLOBAL );
                                        break;
                                    case 7:
                                        disable_interrupts( GLOBAL );
                                        servo_on[ 7 ] = temp_on;
                                        servo_off[ 7 ] = temp_off;
                                        servo_base[ 7 ] = temp_base;
                                        enable_interrupts( GLOBAL );
                                        break;
                                }
                                break;
                            }
                        }
                        break;
                    }
                }
            }
        }
    }
}

/* end of file servo.c */
